<?php
/**
 * WPMR_Checksums_Trait
 *
 * Generated during Phase 2 restructuring
 * This trait contains methods extracted from the original monolithic wpmr.php
 *
 * @package WP_Malware_Removal
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;

}

trait WPMR_Checksums {

	function fails_checksum( $local_file ) {
		if ( $this->is_invalid_file( $local_file ) ) {
			if ( $this->is_scannable_file( $local_file ) ) {
				return true;
			}
			return;
		}
		if ( ! empty( $GLOBALS['WPMR']['regex'] ) ) {
			remove_filter( 'serve_checksums', array( $this, 'get_cached_checksums' ) );
		}

		$hash = @hash_file( 'sha256', $local_file );

		// ensure that for files in core wp dir, only the respective checksums are matched.
		if ( $this->is_in_core_wp_dir( $local_file ) ) {
			$checksums = $GLOBALS['WPMR']['core_checksums'];
		} else { // file not in core wp dir
			$checksums = $GLOBALS['WPMR']['checksums'];
		}

		// $checksums = $GLOBALS['WPMR']['checksums']; // testing if file is shown on rescan
		// For core files, check if the file path exists in checksums AND hash matches
		// For non-core files, just check if hash exists in checksums values
		$checksum_failed = false;
		if ( $this->is_in_core_wp_dir( $local_file ) ) {
			// Core file: must have both the path registered AND matching hash
			$relative_path = $this->get_core_relative_path( $local_file );
			$checksum_failed = ! isset( $checksums[ $relative_path ] ) || $checksums[ $relative_path ] !== $hash;
		} else {
			// Non-core file: just check if hash exists in known checksums
			$checksum_failed = ! in_array( $hash, $checksums );
		}

		if ( $checksum_failed || $this->is_file_blacklisted( $hash ) ) {
			if ( ! $this->is_registered() ) {
				if (
					$this->normalise_path( $local_file ) == $this->normalise_path( trailingslashit( ABSPATH ) . 'wp-config.php' ) ||
					$this->normalise_path( $local_file ) == $this->normalise_path( trailingslashit( ABSPATH ) . '.htaccess' ) ||
					$this->normalise_path( $local_file ) == $this->normalise_path( trailingslashit( WP_CONTENT_DIR ) . 'index.php' ) ||
					$this->normalise_path( $local_file ) == $this->normalise_path( trailingslashit( WP_PLUGIN_DIR ) . 'index.php' ) ||
					$this->normalise_path( $local_file ) == $this->normalise_path( trailingslashit( get_theme_root() ) . 'index.php' )
				) {
					// uncommenting the next line could lead to false negatives
					// return false; // ignore known files that may not have a checksum at all
				}
			}
			return 'missing';
		}
		return;
	}

	function update_cached_checksums( $clean_files ) {
		if ( empty( $clean_files ) ) {
			return;
		}
		$checksums     = $this->get_checksums();
		$new_checksums = array();
		foreach ( $clean_files as $file ) {
			if ( empty( $checksums[ $file ] ) && ! $this->is_invalid_file( $file ) ) {
				$sha256 = @hash_file( 'sha256', $file );
				if ( ! empty( $sha256 ) ) {
					$new_checksums[ $file ] = $sha256;
				}
			}
		}
		$cached_checksums = get_option( 'WPMR_files_checksums_cache' );
		if ( $cached_checksums && is_array( $cached_checksums ) && ! empty( $cached_checksums ) ) {
			update_option( 'WPMR_files_checksums_cache', array_merge( $cached_checksums, $new_checksums ) );
		} else {
			update_option( 'WPMR_files_checksums_cache', $new_checksums );
		}
	}

	function get_checksums_values() {
		$checksums       = $this->get_checksums();
		$checksum_values = array_values( $checksums );
		$checksum_values = array_unique( $checksum_values, SORT_REGULAR );
		return $checksum_values;
	}

	function get_cached_checksums( $checksums ) {
		if ( ! empty( $GLOBALS['WPMR']['debug'] ) ) { // This element doesn't exist on page load, only exists on ajax requests
			return $checksums;
		}
		$cached_checksums = get_option( 'WPMR_files_checksums_cache' );
		if ( $cached_checksums && is_array( $cached_checksums ) && ! empty( $cached_checksums ) ) {
			return array_merge( $checksums, $cached_checksums );
		}
		return $checksums;
	}

	function delete_generated_checksums() {
		delete_option( 'WPMR_files_checksums_cache' );
		delete_option( 'WPMR_db_checksums_cache' );
	}

	function delete_core_checksums( $upgrader, $hook_extra ) {
		$this->flog( '$upgrader' );
		$this->flog( $upgrader );
		$this->flog( '$hook_extra' );
		$this->flog( $hook_extra );
		delete_option( 'WPMR_checksums' );
	}

	function get_checksums( $cached = true ) {
		$this->raise_limits_conditionally();
		$checksums = get_option( 'WPMR_checksums' );
		if ( ! $checksums || ! $cached ) {

			global $wp_version;

			$checksums = $this->sha256_get_core_checksums( $wp_version, $this->get_locale() );
			if ( ! $checksums ) {
				$checksums = array();
				$checksums = $this->sha256_get_core_checksums( $wp_version ); // defaults to en_US
			}
			if ( ! $checksums ) {
				$checksums = array();
			}

			$plugin_checksums = $this->get_plugin_checksums_wpmr();
			if ( $plugin_checksums ) {
				$checksums = array_merge( $checksums, $plugin_checksums );
			} else {
			}

			$theme_checksums = $this->get_theme_checksums();
			if ( $theme_checksums ) {
				$checksums = array_merge( $checksums, $theme_checksums );
			} else {
			}

			if ( $checksums ) {
				update_option( 'WPMR_checksums', $checksums );
			} else {
			}
		}
		return apply_filters( 'serve_checksums', $checksums );
	}

	function sha256_get_core_checksums( $ver = false, $locale = 'en_US' ) {
		$state = $this->get_setting( 'user' );
		$state = $this->encode( $state );
		global $wp_version;
		if ( ! $ver ) {
			$ver = $wp_version;
		}
		$checksum_url = WPMR_SERVER . '?wpmr_action=wpmr_checksum&slug=wordpress&version=' . $ver . '&locale=' . $locale . '&type=core&state=' . $state;
		// $this->flog( __FUNCTION__ . ':' . $checksum_url );

		$core_checksums = array();
		$checksum       = wp_safe_remote_get( $checksum_url, array( 'timeout' => $this->timeout ) );
		if ( is_wp_error( $checksum ) ) {
			return;
		}
		if ( '200' != wp_remote_retrieve_response_code( $checksum ) ) {
			return;
		}
		$checksum = wp_remote_retrieve_body( $checksum );
		$checksum = json_decode( $checksum, true );
		if ( ! is_null( $checksum ) && ! empty( $checksum['files'] ) ) {
			$checksum = $checksum['files'];
			foreach ( $checksum as $file => $checksums ) {
				$core_checksums[ $file ] = $checksums['sha256'];
			}
		}
		return $core_checksums;
	}

	function get_plugin_checksums() {
		$missing          = array();
		$all_plugins      = get_plugins();
		$install_path     = ABSPATH;
		$plugin_checksums = array();
		foreach ( $all_plugins as $key => $value ) {
			if ( false !== strpos( $key, '/' ) ) { // plugin has to be inside a directory. currently drop in plugins are not supported
				$plugin_file  = trailingslashit( dirname( $this->dir ) ) . $key;
				$plugin_file  = str_replace( $install_path, '', $plugin_file );
				$checksum_url = 'https://downloads.wordpress.org/plugin-checksums/' . dirname( $key ) . '/' . $value['Version'] . '.json';
				// $this->flog( __FUNCTION__ . ':' . $checksum_url );
				$checksum = wp_safe_remote_get( $checksum_url, array( 'timeout' => $this->timeout ) );
				if ( is_wp_error( $checksum ) ) {
					continue;
				}
				if ( '200' != wp_remote_retrieve_response_code( $checksum ) ) {
					if ( '404' == wp_remote_retrieve_response_code( $checksum ) ) {
						$missing[ $key ] = array( 'Version' => $value['Version'] );
					}
					continue;
				}
				$checksum = wp_remote_retrieve_body( $checksum );
				$checksum = json_decode( $checksum, true );
				if ( ! is_null( $checksum ) && ! empty( $checksum['files'] ) ) {
					$checksum = $checksum['files'];
					foreach ( $checksum as $file => $checksums ) {
						$plugin_checksums[ trailingslashit( dirname( $plugin_file ) ) . $file ] = $checksums['sha256'];
					}
				}
			} else {
			}
		}
		$extras = $this->get_pro_checksums( $missing );
		if ( $extras ) {
			$plugin_checksums = array_merge( $plugin_checksums, $extras );
		}
		return $plugin_checksums;
	}

	function get_plugin_checksums_wpmr() {
		$missing          = array();
		$all_plugins      = get_plugins();
		$install_path     = ABSPATH;
		$plugin_checksums = array();
		$state            = $this->get_setting( 'user' );
		$state            = $this->encode( $state );
		foreach ( $all_plugins as $key => $value ) {
			if ( false !== strpos( $key, '/' ) ) { // plugin has to be inside a directory. currently drop in plugins are not supported
				$plugin_file  = trailingslashit( dirname( $this->dir ) ) . $key;
				$plugin_file  = str_replace( $install_path, '', $plugin_file );
				$checksum_url = WPMR_SERVER . '?wpmr_action=wpmr_checksum&slug=' . dirname( $key ) . '&version=' . $value['Version'] . '&type=plugin&state=' . $state;
				$checksum     = wp_safe_remote_get( $checksum_url, array( 'timeout' => $this->timeout ) );// $this->timeout ) );
				if ( is_wp_error( $checksum ) ) {
					$missing[ $key ] = array( 'Version' => $value['Version'] );
					continue;
				}
				if ( '200' != wp_remote_retrieve_response_code( $checksum ) ) {
					if ( '404' == wp_remote_retrieve_response_code( $checksum ) ) {
						$missing[ $key ] = array( 'Version' => $value['Version'] );
					}
					continue;
				}

				$checksum = wp_remote_retrieve_body( $checksum );
				$checksum = json_decode( $checksum, true );
				if ( ! is_null( $checksum ) && ! empty( $checksum['files'] ) ) {
					$checksum = $checksum['files'];
					foreach ( $checksum as $file => $checksums ) {
						$plugin_checksums[ trailingslashit( dirname( $plugin_file ) ) . $file ] = $checksums['sha256'];
					}
				} else {
					$missing[ $key ] = array( 'Version' => $value['Version'] );
				}
			} else {
			}
		}
		$extras = $this->get_pro_checksums( $missing );
		if ( $extras ) {
			$plugin_checksums = array_merge( $plugin_checksums, $extras );
		}
		return $plugin_checksums;
	}

	function get_pro_checksums( $missing ) {
		if ( empty( $missing ) || ! $this->is_registered() || ! $this->is_advanced_edition() ) { // can't burden our server
			return;
		}
		$state            = $this->get_setting( 'user' );
		$state            = $this->encode( $state );
		$all_plugins      = $missing;
		$install_path     = ABSPATH;
		$plugin_checksums = array();
		foreach ( $all_plugins as $key => $value ) {
			if ( false !== strpos( $key, '/' ) ) { // plugin has to be inside a directory. currently drop in plugins are not supported
				$plugin_file  = trailingslashit( dirname( $this->dir ) ) . $key;
				$plugin_file  = str_replace( $install_path, '', $plugin_file );
				$checksum_url = WPMR_SERVER . '?wpmr_action=wpmr_checksum&slug=' . dirname( $key ) . '&version=' . $value['Version'] . '&type=plugin&state=' . $state;
				$checksum     = wp_safe_remote_get( $checksum_url, array( 'timeout' => $this->timeout ) );
				if ( is_wp_error( $checksum ) ) {
					continue;
				}
				if ( '200' != wp_remote_retrieve_response_code( $checksum ) ) {
					continue;
				}
				$checksum = wp_remote_retrieve_body( $checksum );
				$checksum = json_decode( $checksum, true );
				if ( ! is_null( $checksum ) && ! empty( $checksum['files'] ) ) {
					$checksum = $checksum['files'];
					foreach ( $checksum as $file => $checksums ) {
						$plugin_checksums[ trailingslashit( dirname( $plugin_file ) ) . $file ] = $checksums['sha256'];
					}
				}
			} else {
			}
		}
		return $plugin_checksums;
	}

	function get_theme_checksums() {
		$all_themes      = wp_get_themes();
		$install_path    = ABSPATH;
		$theme_checksums = array();
		$theme_root      = get_theme_root();
		$state           = $this->get_setting( 'user' );
		$state           = $this->encode( $state );
		foreach ( $all_themes as $key => $value ) {
			$theme_file   = trailingslashit( $theme_root ) . $key;
			$theme_file   = str_replace( $install_path, '', $theme_file );
			$checksum_url = WPMR_SERVER . '?wpmr_action=wpmr_checksum&slug=' . $key . '&version=' . $value['Version'] . '&type=theme&state=' . $state;
			// $this->flog( __FUNCTION__ . ':' . $checksum_url );
			$checksum = wp_safe_remote_get( $checksum_url, array( 'timeout' => $this->timeout ) );
			if ( is_wp_error( $checksum ) ) {
				continue;
			}
			if ( '200' != wp_remote_retrieve_response_code( $checksum ) ) {
				continue;
			}
			$checksum = wp_remote_retrieve_body( $checksum );
			$checksum = json_decode( $checksum, true );
			if ( ! is_null( $checksum ) && ! empty( $checksum['files'] ) ) {
				$checksum = $checksum['files'];
				foreach ( $checksum as $file => $checksums ) {
					$theme_checksums[ trailingslashit( dirname( $theme_file ) ) . $file ] = $checksums['sha256'];
				}
			}
		}
		return $theme_checksums;
	}

	function map_core_checksums( $checksums ) {
		$real_abspath = trailingslashit( $this->normalise_path( ABSPATH ) );
		foreach ( $checksums as $f => $c ) {
			$checksums[ $real_abspath . $f ] = $c;
			unset( $checksums[ $f ] );
		}
		return $checksums;
	}

	function checksums_delete_invalid() {
		$cached_checksums = get_option( 'WPMR_files_checksums_cache' );
		if ( ! $cached_checksums || ! is_array( $cached_checksums ) ) {
			return;
		}
		foreach ( $cached_checksums as $file => $checksum ) {
			if ( $this->is_invalid_file( $file ) ) {
				unset( $cached_checksums[ $file ] );
			}
		}
		update_option( 'WPMR_files_checksums_cache', $cached_checksums );
	}

	function get_db_checksums() {
		return get_option( 'WPMR_db_checksums_cache', array() );
	}
}
