<?php
/**
 * WPMR Pro Operations Trait
 *
 * Handles SaaS control plane communication and local execution of file operations.
 *
 * @package WPMR
 * @since 3.9.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

trait WPMR_API_Operations {

	/**
	 * Public key for RSA signature verification
	 */
	private $saas_public_key = 'MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA4a5vWufgOxDbdeWt9fWRt0L89z4P6kjnL9arr8L/+79zRHIxX8Rfi8nQLy5z9UwsEhGNaqSArR+geApQOdbNmF2bCwOppcBeXFlzRcICadXC04yposn6nkBNUjDQO0qRqNtgF2cM9uzYeL2W7i2N0vy4nPNJo1geny5NVai7GcqAZSh5aAaJLqoQCQ74gZkDDsSVPUkqRwV4V+hj94e3YSTmXrCpmvfDmESfuNvrwFPh/y+zJOIEuM4QWyDcUUElpHrHTE3McB3Qc8gcXsy54wRMrXXhNMoQWufRacidX4fHNBliKu4+r+JB++jYJAWipiCmBoWqZJAtSjWFXOCiAQIDAQAB';

	/**
	 * Request action from SaaS control plane
	 *
	 * @param string $action_type Action type (repair_action, delete_action, whitelist_action)
	 * @param string $file Absolute file path
	 * @return array|WP_Error Response from SaaS or error
	 */
	function request_saas_action( $action_type, $file ) {
		// Collect file context
		if ( ! $this->is_valid_file( $file ) ) {
			return new WP_Error( 'invalid_file', 'Invalid file provided.' );
		}

		$file_sha256 = file_exists( $file ) ? hash_file( 'sha256', $file ) : '';
		$file_type   = $this->determine_file_type( $file );

		if ( 'repair_action' === $action_type && 'unknown' === $file_type ) {
			return new WP_Error( 'unsupported_repair_target', 'Repair is only available for WordPress core, plugin, or theme files.' );
		}
		$file_repo = $this->build_file_repository_context( $file, $file_type );

		$upload_dir = wp_upload_dir();

		// Collect WordPress paths
		$wp_paths = array(
			'ABSPATH'         => ABSPATH,
			'WP_CONTENT_DIR'  => WP_CONTENT_DIR,
			'WP_PLUGIN_DIR'   => WP_PLUGIN_DIR,
			'WPMU_PLUGIN_DIR' => WPMU_PLUGIN_DIR,
			'WP_LANG_DIR'     => defined( 'WP_LANG_DIR' ) ? WP_LANG_DIR : '',
			'WP_TEMP_DIR'     => defined( 'WP_TEMP_DIR' ) ? WP_TEMP_DIR : '',
			'UPLOADS'         => defined( 'UPLOADS' ) ? UPLOADS : '',
			'get_theme_root'  => get_theme_root(),
			'theme_root'      => $this->get_theme_root_paths(),
			'wp_upload_dir'   => isset( $upload_dir['basedir'] ) ? $upload_dir['basedir'] : '',
			'is_subdirectory' => $this->is_subdirectory_install(),
		);

		// Build state

		$state          = array();
		$user           = $this->get_setting( 'user' );
		$lic            = $this->get_setting( 'license_key' );
		$license_status = get_transient( 'WPMR_license_status' );
		$compatibility  = $this->plugin_data;
		if ( $user ) {
			$state['user'] = $user;
		}
		if ( $compatibility ) {
			$state['compatibility'] = $compatibility;
		}
		if ( $lic ) {
			$state['lic'] = $lic;
		}

		$state['license_status'] = $license_status;
		$state['timestamp']      = time();

		$file_payload = array(
			'path'   => $file,
			'sha256' => $file_sha256,
			'kind'   => $file_type,
		);

		if ( ! empty( $file_repo ) ) {
			$file_payload['repo'] = $file_repo;
		}

		// Build request
		$request = array(
			'plugin_version' => $this->plugin_data['Version'],
			'site_url'       => site_url(),
			'home_url'       => home_url(),
			'file'           => $file_payload,
			'wp_paths'       => $wp_paths,
		);

		$state = array_merge( $state, $request );

		$encoded_state = $this->encode( $state );

		// Make request to SaaS
		$response = wp_safe_remote_post(
			WPMR_SERVER . '?wpmr_action=' . $action_type . '&cachebust=' . microtime( 1 ),
			array(
				'body'      => array(
					'state' => $encoded_state,
				),
				'timeout'   => 30,
				'sslverify' => true,
			)
		);

		// Check for errors
		if ( is_wp_error( $response ) ) {
			return new WP_Error( 'saas_unreachable', 'Could not connect to Malcure service. ' . $response->get_error_message() );
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		if ( $response_code !== 200 ) {
			return new WP_Error( 'saas_error', 'Malcure service returned error code: ' . $response_code );
		}

		$body = ltrim( wp_remote_retrieve_body( $response ) );
		if ( stripos( $body, 'null' ) === 0 && strlen( $body ) > 4 ) {
			$body = ltrim( substr( $body, 4 ) );
		}
		$data = json_decode( $body, true );

		if ( empty( $data ) ) {
			return new WP_Error( 'saas_invalid_response', 'Invalid response from Malcure service.' );
		}

		return $data;
	}

	function sanitize_saas_reason_html( $message ) {
		$allowed = array(
			'a'      => array(
				'href'   => array(),
				'target' => array(),
				'rel'    => array(),
			),
			'strong' => array(),
			'em'     => array(),
		);

		return wp_kses( (string) $message, $allowed );
	}

	/**
	 * Validate SaaS response with RSA signature verification
	 *
	 * @param array $response Response from SaaS
	 * @return array Validation result with 'valid' and 'error' keys
	 */
	function validate_saas_response( $response ) {
		// Check required fields
		if ( empty( $response['signature'] ) || empty( $response['action_id'] ) ) {
			return array(
				'valid' => false,
				'error' => 'Invalid response structure from Malcure service.',
			);
		}

		// Check API version compatibility
		if ( empty( $response['api_version'] ) ) {
			return array(
				'valid' => false,
				'error' => 'Missing API version in response.',
			);
		}

		if ( ! function_exists( 'openssl_verify' ) || ! extension_loaded( 'openssl' ) ) {
			return array(
				'valid' => false,
				'error' => 'OpenSSL extension is required to verify responses from the Malcure service. Please enable it on your server.',
			);
		}

		// Extract signature
		$signature = $response['signature'];
		unset( $response['signature'] );

		// Create canonical JSON string
		$json = json_encode( $response, JSON_UNESCAPED_SLASHES );

		// Verify signature
		$public_key = "-----BEGIN PUBLIC KEY-----\n" .
						chunk_split( $this->saas_public_key, 64, "\n" ) .
						'-----END PUBLIC KEY-----';

		$key = openssl_pkey_get_public( $public_key );
		if ( ! $key ) {
			return array(
				'valid' => false,
				'error' => 'Failed to load public key for signature verification.',
			);
		}

		$result = openssl_verify( $json, base64_decode( $signature ), $key, OPENSSL_ALGO_SHA256 );
		if ( is_resource( $key ) ) {
			openssl_free_key( $key );
		}

		if ( $result !== 1 ) {
			return array(
				'valid' => false,
				'error' => 'Signature verification failed. Response may have been tampered with.',
			);
		}

		// Check TTL
		if ( ! empty( $response['ttl_seconds'] ) ) {
			// TTL validation can be added here if timestamp is included in response
		}

		return array(
			'valid' => true,
			'error' => '',
		);
	}

	/**
	 * Execute repair action locally
	 *
	 * @param array  $response Response from SaaS with src URL
	 * @param string $file File path to repair
	 * @return bool|WP_Error True on success, WP_Error on failure
	 */
	function perform_repair_action( $response, $file ) {
		if ( empty( $response['src'] ) ) {
			return new WP_Error( 'no_source', 'No source URL provided for repair.' );
		}

		$perform_action = apply_filters( 'wpmr_perform_edit_action', true, $response, $file, 'repair' );
		if ( ! $perform_action ) {
			return new WP_Error( 'repair_aborted', 'Repair action aborted by filter.' );
		}

		// Fetch file from WordPress.org
		$fetch_response = wp_safe_remote_get(
			$response['src'],
			array(
				'timeout'   => 30,
				'sslverify' => true,
			)
		);

		if ( is_wp_error( $fetch_response ) ) {
			return new WP_Error( 'fetch_failed', 'Failed to fetch file: ' . $fetch_response->get_error_message() );
		}

		$fetch_code = wp_remote_retrieve_response_code( $fetch_response );
		if ( $fetch_code !== 200 ) {
			return new WP_Error( 'fetch_failed', 'Failed to fetch file. HTTP code: ' . $fetch_code );
		}

		$content = wp_remote_retrieve_body( $fetch_response );
		if ( empty( $content ) ) {
			return new WP_Error( 'empty_content', 'Fetched file is empty.' );
		}

		// Calculate SHA256 before
		$sha256_before = file_exists( $file ) ? hash_file( 'sha256', $file ) : '';

		// Write file using WP_Filesystem
		global $wp_filesystem;
		if ( ! function_exists( 'WP_Filesystem' ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}
		WP_Filesystem();

		if ( ! $wp_filesystem->put_contents( $file, $content, FS_CHMOD_FILE ) ) {
			return new WP_Error( 'write_failed', 'Failed to write file.' );
		}

		// Calculate SHA256 after
		$sha256_after = hash_file( 'sha256', $file );

		// Log operation
		$this->log_event(
			'file_repaired',
			array(
				'file'          => $file,
				'action_id'     => $response['action_id'],
				'sha256_before' => $sha256_before,
				'sha256_after'  => $sha256_after,
			)
		);

		return true;
	}

	/**
	 * Execute delete action locally
	 *
	 * @param array  $response Response from SaaS
	 * @param string $file File path to delete
	 * @return bool|WP_Error True on success, WP_Error on failure
	 */
	function perform_delete_action( $response, $file ) {
		// Double-check file is deletable
		if ( ! $this->is_deletable( $file ) ) {
			return new WP_Error( 'not_deletable', 'File cannot be deleted (critical system file).' );
		}

		if ( ! file_exists( $file ) ) {
			return new WP_Error( 'file_not_found', 'File does not exist.' );
		}

		$perform_action = apply_filters( 'wpmr_perform_edit_action', true, $response, $file, 'delete' );
		if ( ! $perform_action ) {
			return new WP_Error( 'delete_aborted', 'Delete action aborted by filter.' );
		}

		// Calculate SHA256 before deletion
		$sha256_before = hash_file( 'sha256', $file );

		// Delete file using WP_Filesystem
		global $wp_filesystem;
		if ( ! function_exists( 'WP_Filesystem' ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}
		WP_Filesystem();

		if ( ! $wp_filesystem->delete( $file, false, 'f' ) ) {
			return new WP_Error( 'delete_failed', 'Failed to delete file.' );
		}

		// Log operation
		$this->log_event(
			'file_deleted',
			array(
				'file'      => $file,
				'action_id' => $response['action_id'],
				'sha256'    => $sha256_before,
			)
		);

		return true;
	}

	/**
	 * Execute whitelist action locally
	 *
	 * @param array  $response Response from SaaS
	 * @param string $file File path to whitelist
	 * @return bool|WP_Error True on success, WP_Error on failure
	 */
	function perform_whitelist_action( $response, $file ) {
		if ( ! file_exists( $file ) ) {
			return new WP_Error( 'file_not_found', 'File does not exist.' );
		}

		$perform_action = apply_filters( 'wpmr_perform_edit_action', true, $response, $file, 'whitelist' );
		if ( ! $perform_action ) {
			return new WP_Error( 'whitelist_aborted', 'Whitelist action aborted by filter.' );
		}

		$whitelist = $this->get_setting( 'whitelist' );
		if ( ! is_array( $whitelist ) ) {
			$whitelist = array();
		}

		$file_sha256 = @hash_file( 'sha256', $file );
		if ( ! $file_sha256 ) {
			return new WP_Error( 'whitelist_failed', 'Can\'t whitelist. File: ' . $file );
		}

		$whitelist[ $file ] = $file_sha256;
		$this->update_setting( 'whitelist', $whitelist );

		// Log operation
		$this->log_event(
			'file_whitelisted',
			array(
				'file'      => $file,
				'action_id' => $response['action_id'],
				'sha256'    => $file_sha256,
			)
		);

		return true;
	}

	/**
	 * Gather theme root directories including registered directories.
	 *
	 * @return array
	 */
	private function get_theme_root_paths() {
		$roots        = array();
		$default_root = get_theme_root();
		if ( $default_root ) {
			$roots[] = wp_normalize_path( $default_root );
		}

		global $_wp_theme_directories;
		if ( ! empty( $_wp_theme_directories ) && is_array( $_wp_theme_directories ) ) {
			foreach ( $_wp_theme_directories as $dir ) {
				$roots[] = wp_normalize_path( $dir );
			}
		}

		return array_values( array_unique( $roots ) );
	}

	/**
	 * Determine file type (core, plugin, theme, unknown)
	 *
	 * @param string $file Absolute file path
	 * @return string File type
	 */
	function determine_file_type( $file ) {

		if ( ! $this->is_repairable( $file ) ) {
			return 'unknown';
		}

		if ( strpos( $file, WP_PLUGIN_DIR ) !== false ) { // file is inside plugins directory
			if ( ! function_exists( 'get_plugins' ) ) {
				require_once ABSPATH . 'wp-admin/includes/plugin.php';
			}

			$plugins = get_plugins();
			foreach ( $plugins as $plugin_file => $plugin_data ) {
				$plugin_dir = dirname( $plugin_file );
				if ( '' === $plugin_dir || '.' === $plugin_dir ) {
					continue; // skip single-file plugins without directory for now
				}

				$plugin_path = wp_normalize_path( trailingslashit( WP_PLUGIN_DIR ) . trailingslashit( $plugin_dir ) );
				if ( strpos( wp_normalize_path( $file ), $plugin_path ) === 0 ) {
					return 'plugin';
				}
			}
		}

		$themes = wp_get_themes();
		foreach ( $themes as $tk => $tv ) {
			if ( strpos( wp_normalize_path( $file ), wp_normalize_path( get_theme_root( $tk ) . DIRECTORY_SEPARATOR . $tk ) ) !== false ) {
				return 'theme';
			}
		}

		remove_filter( 'serve_checksums', array( $this, 'get_cached_checksums' ) );

		$checksums = $this->get_checksums();
		if ( array_key_exists( $this->normalise_path( $file ), $checksums ) ) {
			return 'core';
		}

		return 'unknown';
	}

	/**
	 * Build repository context for the requested file.
	 *
	 * @param string $file Absolute file path.
	 * @param string $file_type Classified file type.
	 * @return array Repository context metadata.
	 */
	private function build_file_repository_context( $file, $file_type ) {
		$file_type = strtolower( $file_type );
		$file      = wp_normalize_path( $file );

		switch ( $file_type ) {
			case 'core':
				return $this->get_core_repository_context( $file );
			case 'plugin':
				return $this->get_plugin_repository_context( $file );
			case 'theme':
				return $this->get_theme_repository_context( $file );
		}

		return array();
	}

	/**
	 * Repository context for WordPress core files.
	 *
	 * @param string $file Normalized absolute file path.
	 * @return array
	 */
	private function get_core_repository_context( $file ) {
		$abspath = wp_normalize_path( trailingslashit( ABSPATH ) );
		if ( strpos( $file, $abspath ) !== 0 ) {
			return array();
		}

		$relative = $this->sanitize_relative_path( substr( $file, strlen( $abspath ) ) );
		if ( '' === $relative ) {
			return array();
		}

		$version = get_bloginfo( 'version' );
		if ( empty( $version ) && isset( $GLOBALS['wp_version'] ) ) {
			$version = $GLOBALS['wp_version'];
		}

		return array(
			'type'          => 'core',
			'slug'          => 'wordpress',
			'version'       => $version,
			'relative_path' => $relative,
		);
	}

	/**
	 * Repository context for plugin files.
	 *
	 * @param string $file Normalized absolute file path.
	 * @return array
	 */
	private function get_plugin_repository_context( $file ) {
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$plugins          = get_plugins();
		$plugins_base_dir = wp_normalize_path( trailingslashit( WP_PLUGIN_DIR ) );

		foreach ( $plugins as $plugin_file => $plugin_data ) {
			$plugin_abs_path = wp_normalize_path( $plugins_base_dir . $plugin_file );
			$plugin_dir_path = wp_normalize_path( trailingslashit( dirname( $plugin_abs_path ) ) );

			if ( strpos( $file, $plugin_dir_path ) !== 0 ) {
				continue;
			}

			$relative_path = $this->sanitize_relative_path( substr( $file, strlen( $plugin_dir_path ) ) );
			$parts         = explode( '/', $plugin_file );
			$plugin_slug   = ( count( $parts ) > 1 ) ? $parts[0] : basename( $plugin_file, '.php' );
			$version       = isset( $plugin_data['Version'] ) ? $plugin_data['Version'] : '';

			return array(
				'type'          => 'plugin',
				'slug'          => sanitize_key( $plugin_slug ),
				'version'       => $version,
				'relative_path' => $relative_path,
			);
		}

		$mu_base_dir = wp_normalize_path( trailingslashit( WPMU_PLUGIN_DIR ) );
		if ( ! empty( $mu_base_dir ) && strpos( $file, $mu_base_dir ) === 0 ) {
			$relative_path = $this->sanitize_relative_path( substr( $file, strlen( $mu_base_dir ) ) );
			$slug          = $relative_path;
			$slash_pos     = strpos( $relative_path, '/' );
			if ( false !== $slash_pos ) {
				$slug = substr( $relative_path, 0, $slash_pos );
			}

			return array(
				'type'          => 'plugin',
				'slug'          => sanitize_key( $slug ),
				'version'       => '',
				'relative_path' => $relative_path,
			);
		}

		return array();
	}

	/**
	 * Repository context for theme files.
	 *
	 * @param string $file Normalized absolute file path.
	 * @return array
	 */
	private function get_theme_repository_context( $file ) {
		if ( ! function_exists( 'wp_get_themes' ) ) {
			require_once ABSPATH . 'wp-includes/theme.php';
		}

		$themes = wp_get_themes();

		foreach ( $themes as $stylesheet => $theme ) {
			$theme_dir = wp_normalize_path( trailingslashit( $theme->get_stylesheet_directory() ) );
			if ( strpos( $file, $theme_dir ) !== 0 ) {
				continue;
			}

			$relative_path = $this->sanitize_relative_path( substr( $file, strlen( $theme_dir ) ) );
			$version       = $theme->get( 'Version' );

			return array(
				'type'          => 'theme',
				'slug'          => sanitize_key( $stylesheet ),
				'version'       => $version,
				'relative_path' => $relative_path,
			);
		}

		return array();
	}

	/**
	 * Normalize and sanitize a relative path segment for transport.
	 *
	 * @param string $path Path fragment.
	 * @return string
	 */
	private function sanitize_relative_path( $path ) {
		$path     = wp_normalize_path( $path );
		$segments = explode( '/', $path );
		$clean    = array();

		foreach ( $segments as $segment ) {
			if ( '' === $segment || '.' === $segment || '..' === $segment ) {
				continue;
			}
			$clean[] = $segment;
		}

		return implode( '/', $clean );
	}

	/**
	 * Detect if WordPress is installed in a subdirectory
	 *
	 * @return bool True if subdirectory install
	 */
	function is_subdirectory_install() {
		if ( site_url() !== home_url() ) {
			return true;
		}

		if ( ! empty( $_SERVER['DOCUMENT_ROOT'] ) ) {
			$doc_root = trailingslashit( $_SERVER['DOCUMENT_ROOT'] );
			$abspath  = trailingslashit( ABSPATH );
			if ( $abspath !== $doc_root ) {
				return true;
			}
		}

		return false;
	}
}
